<?php

/**
 * Füllt die Tabelle "runs" basierend auf den Einträgen in "schedule" und dem zugehörigen Heatsystem.
 * Verwendet das "snake" oder "aal" System gemäß den IWWF Cablewakeboard World Rules.
 * 
 * @param mysqli $conn Die MySQLi-Datenbankverbindung.
 */
function populateRunsTable($conn) {
	// 1. Abrufen aller relevanten Schedule-Einträge und der zugehörigen Heatsystem-Details
	$scheduleQuery = "
		SELECT s.id AS schedule_id, s.comp, s.event, s.category, s.heatsystem, s.order, h.capacity, h.runs, h.system
		FROM schedule s
		JOIN heatsystem h ON s.heatsystem = h.id
	";
	
	$scheduleResult = $conn->query($scheduleQuery);

	if ($scheduleResult->num_rows === 0) {
		echo "Keine relevanten Schedule-Einträge gefunden.\n";
		return;
	}

	// 2. Für jeden Schedule-Eintrag die Gruppen und Läufe in der "runs"-Tabelle anlegen
	while ($scheduleRow = $scheduleResult->fetch_assoc()) {
		$scheduleId = $scheduleRow['schedule_id'];
		$capacity = $scheduleRow['capacity'];
		$numRuns = $scheduleRow['runs'];
		$system = $scheduleRow['system'];
		
		// Beispielhafte Athletenliste für Demonstrationszwecke
		// In einer realen Anwendung würden hier die Athleten für die entsprechende Competition und Kategorie abgerufen
		$athletes = getAthletesForSchedule($conn, $scheduleId); // Funktion, die die Athleten abruft

		// Verteilen der Athleten in Gruppen basierend auf dem System
		if ($system === 'snake') {
			$groups = distributeAthletesSnake($athletes, $capacity);
		} elseif ($system === 'aal') {
			$groups = distributeAthletesAal($athletes, $capacity);
		} else {
			echo "Unbekanntes System '$system' für Schedule ID $scheduleId.\n";
			continue;
		}

		// 3. Füllen der "runs"-Tabelle für jede Gruppe und jeden Lauf
		foreach ($groups as $groupIndex => $groupMembers) {
			for ($run = 1; $run <= $numRuns; $run++) {
				foreach ($groupMembers as $order => $athleteId) {
					$insertRunQuery = "
						INSERT INTO runs (schedule, group, `order`, run, athlete, origin, status, timestamp)
						VALUES (?, ?, ?, ?, ?, NULL, 0, CURRENT_TIMESTAMP)
					";
					$insertStmt = $conn->prepare($insertRunQuery);
					$group = $groupIndex + 1; // Gruppenindex in menschlicher Zählweise
					$insertStmt->bind_param("iiiii", $scheduleId, $group, $order + 1, $run, $athleteId);
					$insertStmt->execute();
					$insertStmt->close();
				}
			}
		}
	}
}

/**
 * Verteilt Athleten nach dem Snake-System.
 * 
 * @param array $athletes Liste der Athleten-IDs, sortiert nach Ranking.
 * @param int $capacity Maximalanzahl an Athleten pro Gruppe.
 * @return array Gruppen mit Athleten, die im Snake-System verteilt wurden.
 */
function distributeAthletesSnake($athletes, $capacity) {
	$numGroups = ceil(count($athletes) / $capacity);
	$groups = array_fill(0, $numGroups, []);
	$index = 0;
	$reverse = false;

	foreach ($athletes as $athlete) {
		$groups[$index][] = $athlete;
		if ($reverse) {
			$index--;
			if ($index < 0) {
				$index = 0;
				$reverse = false;
			}
		} else {
			$index++;
			if ($index >= $numGroups) {
				$index = $numGroups - 1;
				$reverse = true;
			}
		}
	}

	return $groups;
}

/**
 * Verteilt Athleten nach dem AAL-System.
 * 
 * @param array $athletes Liste der Athleten-IDs, sortiert nach Ranking.
 * @param int $capacity Maximalanzahl an Athleten pro Gruppe.
 * @return array Gruppen mit Athleten, die im AAL-System verteilt wurden.
 */
function distributeAthletesAal($athletes, $capacity) {
	$numGroups = ceil(count($athletes) / $capacity);
	$groups = array_fill(0, $numGroups, []);

	for ($i = 0; $i < count($athletes); $i++) {
		$groupIndex = $i % $numGroups;
		$groups[$groupIndex][] = $athletes[$i];
	}

	return $groups;
}

?>